"""Does general configuration parsing; used by other classes for their configuration."""

import os
import warnings
from configparser import ConfigParser


class ConfigParameter(object):
    """Contains information about one configuration item."""

    def __init__(self, name, value_type, default=None):
        self.name = name
        self.value_type = value_type
        self.default = default

    def __repr__(self):
        if self.default is None:
            return f"ConfigParameter({self.name!r}, {self.value_type!r})"
        return f"ConfigParameter({self.name!r}, {self.value_type!r}, {self.default!r})"

    def parse(self, section, config_parser):
        if int == self.value_type:
            return config_parser.getint(section, self.name)
        if bool == self.value_type:
            return config_parser.getboolean(section, self.name)
        if float == self.value_type:
            return config_parser.getfloat(section, self.name)
        if list == self.value_type:
            v = config_parser.get(section, self.name)
            return v.split(" ")
        if str == self.value_type:
            return config_parser.get(section, self.name)

        raise RuntimeError(f"Unexpected configuration type: {self.value_type!r}")

    def interpret(self, config_dict):
        """
        Converts the config_parser output into the proper type,
        supplies defaults if available and needed, and checks for some errors.
        """
        value = config_dict.get(self.name)
        if value is None:
            if self.default is None:
                raise RuntimeError("Missing configuration item: " + self.name)
            else:
                warnings.warn(
                    f"Using default {self.default!r} for '{self.name!s}'",
                    DeprecationWarning,
                )
                if (str != self.value_type) and isinstance(
                    self.default, self.value_type
                ):
                    return self.default
                else:
                    value = self.default

        try:
            if str == self.value_type:
                return str(value)
            if int == self.value_type:
                return int(value)
            if bool == self.value_type:
                if value.lower() == "true":
                    return True
                elif value.lower() == "false":
                    return False
                else:
                    raise RuntimeError(self.name + " must be True or False")
            if float == self.value_type:
                return float(value)
            if list == self.value_type:
                return value.split(" ")
        except Exception:
            raise RuntimeError(
                f"Error interpreting config item '{self.name}' with value {value!r} and type {self.value_type}"
            )

        raise RuntimeError("Unexpected configuration type: " + repr(self.value_type))

    def format(self, value):
        if list == self.value_type:
            return " ".join(value)
        return str(value)


def write_pretty_params(f, config, params):
    param_names = [p.name for p in params]
    longest_name = max(len(name) for name in param_names)
    param_names.sort()
    params = dict((p.name, p) for p in params)

    for name in param_names:
        p = params[name]
        f.write(f"{p.name.ljust(longest_name)} = {p.format(getattr(config, p.name))}\n")


class UnknownConfigItemError(NameError):
    """Error for unknown configuration option - partially to catch typos."""

    pass


class DefaultClassConfig(object):
    """
    Replaces at least some boilerplate configuration code
    for reproduction, species_set, and stagnation classes.
    """

    def __init__(self, param_dict, param_list):
        self._params = param_list
        param_list_names = []
        for p in param_list:
            setattr(self, p.name, p.interpret(param_dict))
            param_list_names.append(p.name)
        unknown_list = [x for x in param_dict if x not in param_list_names]
        if unknown_list:
            if len(unknown_list) > 1:
                raise UnknownConfigItemError(
                    "Unknown configuration items:\n" + "\n\t".join(unknown_list)
                )
            raise UnknownConfigItemError(
                f"Unknown configuration item {unknown_list[0]!s}"
            )

    @classmethod
    def write_config(cls, f, config):
        # pylint: disable=protected-access
        write_pretty_params(f, config, config._params)


class Config(object):
    """A container for user-configurable parameters of NEAT."""

    __params = [
        ConfigParameter("pop_size", int),
        ConfigParameter("fitness_criterion", str),
        ConfigParameter("fitness_threshold", float),
        ConfigParameter("reset_on_extinction", bool),
        ConfigParameter("no_fitness_termination", bool, False),
        ConfigParameter("gen", int),
        ConfigParameter("merge_num", int),
        ConfigParameter("pcdtype", str),
        ConfigParameter("test_mode", bool),
        ConfigParameter("novelty_threshold", float),
        ConfigParameter("archive_size", int),
        ConfigParameter("knn", int),
        ConfigParameter("density", int),
        ConfigParameter("global_rate", float),
    ]

    def __init__(
        self,
        genome_type,
        reproduction_type,
        species_set_type,
        stagnation_type,
        filename,
        config_information=None,
    ):
        # Check that the provided types have the required methods.
        assert hasattr(genome_type, "parse_config")
        assert hasattr(reproduction_type, "parse_config")
        assert hasattr(species_set_type, "parse_config")
        assert hasattr(stagnation_type, "parse_config")

        self.genome_type = genome_type
        self.reproduction_type = reproduction_type
        self.species_set_type = species_set_type
        self.stagnation_type = stagnation_type
        self.config_information = config_information

        if not os.path.isfile(filename):
            raise Exception("No such config file: " + os.path.abspath(filename))

        parameters = ConfigParser()
        with open(filename) as f:
            parameters.read_file(f)

        # NEAT configuration
        if not parameters.has_section("NEAT"):
            raise RuntimeError("'NEAT' section not found in NEAT configuration file.")

        param_list_names = []
        for p in self.__params:
            if p.default is None:
                setattr(self, p.name, p.parse("NEAT", parameters))
            else:
                try:
                    setattr(self, p.name, p.parse("NEAT", parameters))
                except Exception:
                    setattr(self, p.name, p.default)
                    warnings.warn(
                        f"Using default {p.default!r} for '{p.name!s}'",
                        DeprecationWarning,
                    )
            param_list_names.append(p.name)
        param_dict = dict(parameters.items("NEAT"))
        unknown_list = [x for x in param_dict if x not in param_list_names]
        if unknown_list:
            if len(unknown_list) > 1:
                raise UnknownConfigItemError(
                    "Unknown (section 'NEAT') configuration items:\n"
                    + "\n\t".join(unknown_list)
                )
            raise UnknownConfigItemError(
                f"Unknown (section 'NEAT') configuration item {unknown_list[0]!s}"
            )

        # Parse type sections.
        genome_dict = dict(parameters.items(genome_type.__name__))
        self.genome_config = genome_type.parse_config(genome_dict)

        species_set_dict = dict(parameters.items(species_set_type.__name__))
        self.species_set_config = species_set_type.parse_config(species_set_dict)

        stagnation_dict = dict(parameters.items(stagnation_type.__name__))
        self.stagnation_config = stagnation_type.parse_config(stagnation_dict)

        reproduction_dict = dict(parameters.items(reproduction_type.__name__))
        self.reproduction_config = reproduction_type.parse_config(reproduction_dict)

    def save(self, filename):
        with open(filename, "w") as f:
            f.write(
                "# The `NEAT` section specifies parameters particular to the NEAT algorithm\n"
            )
            f.write("# or the experiment itself.  This is the only required section.\n")
            f.write("[NEAT]\n")
            write_pretty_params(f, self, self.__params)

            f.write(f"\n[{self.genome_type.__name__}]\n")
            self.genome_type.write_config(f, self.genome_config)

            f.write(f"\n[{self.species_set_type.__name__}]\n")
            self.species_set_type.write_config(f, self.species_set_config)

            f.write(f"\n[{self.stagnation_type.__name__}]\n")
            self.stagnation_type.write_config(f, self.stagnation_config)

            f.write(f"\n[{self.reproduction_type.__name__}]\n")
            self.reproduction_type.write_config(f, self.reproduction_config)
